<?php
// Konstanty
define('SPEED_OF_LIGHT', 299792458); // Rychlost světla v m/s

// Funkce pro výpočet vlnové délky
function calculateWavelength($frequency) {
    // Převod frekvence z MHz na Hz
    $frequencyHz = $frequency * 1000000;

    // Výpočet vlnové délky v metrech
    $wavelengthMeters = SPEED_OF_LIGHT / $frequencyHz;

    return $wavelengthMeters;
}

// Funkce pro určení typu pásma
function determineBandType($frequency) {
    // Frekvence je v MHz, převedeme na správné jednotky pro porovnání
    if ($frequency < 0.00003) {
        return "ELF (Extremely Low Frequency)";
    } else if ($frequency < 0.0003) {
        return "SLF (Super Low Frequency)";
    } else if ($frequency < 0.003) {
        return "ULF (Ultra Low Frequency)";
    } else if ($frequency < 0.03) {
        return "VLF (Very Low Frequency)";
    } else if ($frequency < 0.3) {
        return "LF (Low Frequency)";
    } else if ($frequency < 3) {
        return "MF (Medium Frequency)";
    } else if ($frequency < 30) {
        return "HF (High Frequency)";
    } else if ($frequency < 300) {
        return "VHF (Very High Frequency)";
    } else if ($frequency < 3000) {
        return "UHF (Ultra High Frequency)";
    } else if ($frequency < 30000) {
        return "SHF (Super High Frequency)";
    } else if ($frequency < 300000) {
        return "EHF (Extremely High Frequency)";
    } else {
        return "THF (Tremendously High Frequency)";
    }
}

// Funkce pro určení názvu vlnového pásma
function determineWavelengthBand($wavelengthMeters) {
    // Standardní radioamatérská pásma v metrech a centimetrech
    $standardBands = [
        // Pásma v metrech
        ["name" => "160m", "min" => 150, "max" => 170],
        ["name" => "80m", "min" => 75, "max" => 85],
        ["name" => "60m", "min" => 55, "max" => 65],
        ["name" => "40m", "min" => 38, "max" => 43],
        ["name" => "30m", "min" => 28, "max" => 32],
        ["name" => "20m", "min" => 19, "max" => 22],
        ["name" => "17m", "min" => 16, "max" => 18.5],
        ["name" => "15m", "min" => 14, "max" => 16],
        ["name" => "12m", "min" => 11.5, "max" => 13],
        ["name" => "10m", "min" => 9.5, "max" => 11],
        ["name" => "6m", "min" => 5, "max" => 7],
        ["name" => "4m", "min" => 3.5, "max" => 4.5],
        ["name" => "2m", "min" => 1.8, "max" => 2.2],
        ["name" => "1.25m", "min" => 1.1, "max" => 1.4],

        // Pásma v centimetrech (wavelengthMeters < 1)
        ["name" => "70cm", "min" => 0.65, "max" => 0.75],
        ["name" => "33cm", "min" => 0.3, "max" => 0.36],
        ["name" => "23cm", "min" => 0.21, "max" => 0.25],
        ["name" => "13cm", "min" => 0.12, "max" => 0.14],
        ["name" => "9cm", "min" => 0.08, "max" => 0.1],
        ["name" => "6cm", "min" => 0.05, "max" => 0.07],
        ["name" => "3cm", "min" => 0.028, "max" => 0.032],
        ["name" => "1.25cm", "min" => 0.0115, "max" => 0.0135],
        ["name" => "6mm", "min" => 0.0055, "max" => 0.0065],
        ["name" => "4mm", "min" => 0.0035, "max" => 0.0045],
        ["name" => "2.5mm", "min" => 0.0023, "max" => 0.0028]
    ];

    // Nalezení odpovídajícího standardního pásma
    foreach ($standardBands as $band) {
        if ($wavelengthMeters >= $band["min"] && $wavelengthMeters <= $band["max"]) {
            return $band["name"] . " pásmo";
        }
    }

    // Pokud není nalezeno žádné standardní pásmo, vrátit přesnější popis
    if ($wavelengthMeters >= 1) {
        // Pro vlnové délky >= 1m, vrátit v metrech
        return round($wavelengthMeters) . "m pásmo";
    } else if ($wavelengthMeters >= 0.01) {
        // Pro vlnové délky mezi 1cm a 1m, vrátit v centimetrech
        return round($wavelengthMeters * 100) . "cm pásmo";
    } else {
        // Pro vlnové délky < 1cm, vrátit v milimetrech
        return round($wavelengthMeters * 1000) . "mm pásmo";
    }
}

// Funkce pro odhad dosahu
function estimateRange($frequency, $power, $terrain) {
    // Faktory útlumu terénu (přibližné hodnoty založené na reálných zkušenostech)
    $terrainFactors = [
        'open' => 1.0,      // Otevřený terén - referenční hodnota
        'suburban' => 0.8,  // Příměstská oblast - mírný útlum
        'urban' => 0.6,     // Městská oblast - významný útlum
        'forest' => 0.7,    // Les - střední útlum
        'mountains' => 0.5, // Hornatý terén - vysoký útlum, ale může pomoci při šíření na větší vzdálenosti
        'water' => 1.3      // Vodní plocha - výrazně lepší šíření než otevřený terén
    ];

    // Realističtější výpočet dosahu založený na empirických datech a zkušenostech radioamatérů
    // Rozdělení podle frekvenčních pásem
    if ($frequency < 3) { // LF/MF pásma
        // Nízké frekvence mají velký dosah, zejména v noci
        $baseRange = 800 * pow($power / 100, 0.4);
    } else if ($frequency < 30) { // HF pásmo
        // HF pásmo má velký dosah díky ionosférickému šíření
        // Dosah závisí na denní době, ročním období a sluneční aktivitě
        $baseRange = 2000 * pow($power / 100, 0.3);
    } else if ($frequency < 300) { // VHF pásmo
        // VHF pásmo má přímočaré šíření, ale za dobrých podmínek může dosáhnout velkých vzdáleností
        // Při inverzi nebo troposférickém šíření může být dosah mnohem větší
        $baseRange = 100 * pow($power / 50, 0.25);
    } else if ($frequency < 3000) { // UHF pásmo
        // UHF pásmo může za dobrých podmínek dosáhnout překvapivě velkých vzdáleností
        // Zejména při použití směrových antén nebo při troposférickém šíření
        $baseRange = 120 * pow($power / 50, 0.2);
    } else { // SHF a vyšší pásma
        // Mikrovlnné frekvence mohou mít dobrý dosah při použití směrových antén
        $baseRange = 50 * pow($power / 50, 0.15);
    }

    // Aplikace faktoru terénu
    $terrainFactor = isset($terrainFactors[$terrain]) ? $terrainFactors[$terrain] : 0.7;

    // Výpočet konečného dosahu
    $range = $baseRange * $terrainFactor;

    // Vrácení dosahu zaokrouhleného na jedno desetinné místo
    return round($range * 10) / 10;
}

// Zpracování formuláře
$frequency = 145; // Výchozí hodnota
$power = 50;      // Výchozí hodnota
$terrain = 'open'; // Výchozí hodnota

// Pokud byl formulář odeslán, použijeme hodnoty z formuláře
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $frequency = isset($_POST["frequency"]) ? floatval($_POST["frequency"]) : $frequency;
    $power = isset($_POST["power"]) ? floatval($_POST["power"]) : $power;
    $terrain = isset($_POST["terrain"]) ? $_POST["terrain"] : $terrain;
}

// Výpočet výsledků
$wavelengthMeters = calculateWavelength($frequency);
$wavelengthMm = $wavelengthMeters * 1000;
$wavelengthCm = $wavelengthMeters * 100;
$wavelengthBand = determineWavelengthBand($wavelengthMeters);
$bandType = determineBandType($frequency);
$estimatedRange = estimateRange($frequency, $power, $terrain);

// Funkce pro kontrolu, zda je terén vybrán
function isTerrainSelected($terrainOption, $currentTerrain) {
    return $terrainOption === $currentTerrain ? 'selected' : '';
}

// Nastavení tmavého režimu jako výchozího
$darkMode = true;
?>

<!DOCTYPE html>
<html lang="cs">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Frekvenční vizualizér - AllHam</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Custom CSS -->
    <link rel="stylesheet" href="css/styles.css">
    <!-- Dark Mode CSS -->
    <link rel="stylesheet" href="css/dark-mode.css">
    <!-- Form Styles CSS -->
    <link rel="stylesheet" href="css/form-styles.css">
    <style>
        /* Tmavý režim jako výchozí */
        body {
            background-color: #121212;
            color: #e0e0e0;
        }

        .card {
            background-color: #1e1e1e;
            border-color: #333;
        }

        .card-body {
            color: #e0e0e0;
        }

        .form-control, .form-select {
            background-color: #2a2a2a;
            border-color: #444;
            color: #e0e0e0;
        }

        .form-control:focus, .form-select:focus {
            background-color: #333;
            color: #fff;
            border-color: #0d6efd;
            box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
        }

        /* Styly pro výsledky */
        .result-item {
            margin-bottom: 1.5rem;
            padding: 1rem;
            background-color: #2a2a2a;
            border-radius: 0.5rem;
            border-left: 4px solid #0d6efd;
        }

        .result-item h4 {
            color: #0d6efd;
            margin-bottom: 0.5rem;
        }

        .result-item p {
            font-size: 1.2rem;
            margin-bottom: 0.25rem;
        }

        /* Vizualizace vlny */
        #wave-visualization {
            width: 100%;
            height: 150px;
            position: relative;
            overflow: hidden;
            border-radius: 4px;
            border: 1px solid #444;
            background-color: #2a2a2a;
        }

        #wave-info {
            position: absolute;
            top: 10px;
            left: 10px;
            font-size: 12px;
            background-color: rgba(255,255,255,0.1);
            padding: 5px;
            border-radius: 3px;
            color: #e0e0e0;
        }

        /* Tabulka */
        .table-dark {
            background-color: #2a2a2a;
            color: #e0e0e0;
        }

        .table-dark th {
            background-color: #0277bd;
            color: white;
            border-color: #0288d1;
        }

        .table-dark td {
            border-color: #444;
        }
    </style>
</head>
<body class="dark-mode">
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="index.php">AllHam</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php"><i class="fas fa-home"></i> Domů</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="frequency.php"><i class="fas fa-broadcast-tower"></i> Frekvenční vizualizér</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="prefix.php"><i class="fas fa-globe"></i> Vyhledávání prefixů</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="wavelength.php"><i class="fas fa-wave-square"></i> Délka vlny</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="callsign.php"><i class="fas fa-microphone"></i> Hláskovač</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="antenna.php"><i class="fas fa-satellite-dish"></i> Výpočet antény</a>
                    </li>
                </ul>

            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div id="content">
            <h1 class="text-center mb-4 text-light">Frekvenční vizualizér</h1>

            <div class="row">
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h3 class="card-title">Parametry</h3>
                        </div>
                        <div class="card-body">
                            <form id="radio-form" method="post" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>">
                                <div class="mb-3">
                                    <label for="frequency" class="form-label">Frekvence (MHz)</label>
                                    <input type="number" class="form-control" id="frequency" name="frequency" placeholder="Např. 145" value="<?php echo $frequency; ?>" required step="any">
                                </div>

                                <div class="mb-3">
                                    <label for="power" class="form-label">Výkon (W)</label>
                                    <input type="number" class="form-control" id="power" name="power" placeholder="Např. 50" value="<?php echo $power; ?>" required step="any">
                                </div>

                                <div class="mb-3">
                                    <label for="terrain" class="form-label">Terén</label>
                                    <select class="form-select" id="terrain" name="terrain" required>
                                        <option value="open" <?php echo isTerrainSelected('open', $terrain); ?>>Otevřený terén</option>
                                        <option value="suburban" <?php echo isTerrainSelected('suburban', $terrain); ?>>Příměstská oblast</option>
                                        <option value="urban" <?php echo isTerrainSelected('urban', $terrain); ?>>Městská oblast</option>
                                        <option value="forest" <?php echo isTerrainSelected('forest', $terrain); ?>>Les</option>
                                        <option value="mountains" <?php echo isTerrainSelected('mountains', $terrain); ?>>Hory</option>
                                        <option value="water" <?php echo isTerrainSelected('water', $terrain); ?>>Vodní plocha</option>
                                    </select>
                                </div>

                                <button type="submit" class="btn btn-primary">Vypočítat</button>
                            </form>
                        </div>
                    </div>
                </div>

                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header bg-success text-white">
                            <h3 class="card-title">Výsledky</h3>
                        </div>
                        <div class="card-body">
                            <div id="results">
                                <div class="result-item">
                                    <h4>Délka vlny</h4>
                                    <p id="wavelength-mm"><?php echo number_format($wavelengthMm, 2); ?> mm</p>
                                    <p id="wavelength-cm"><?php echo number_format($wavelengthCm, 2); ?> cm</p>
                                    <p id="wavelength-band"><?php echo $wavelengthBand; ?></p>
                                </div>

                                <div class="result-item">
                                    <h4>Typ pásma</h4>
                                    <p id="band-type"><?php echo $bandType; ?></p>
                                </div>

                                <div class="result-item">
                                    <h4>Odhadovaný dosah</h4>
                                    <p id="estimated-range"><?php echo $estimatedRange; ?> km</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="card mt-3">
                        <div class="card-header bg-info text-white">
                            <h3 class="card-title">Vizualizace vlny</h3>
                        </div>
                        <div class="card-body">
                            <p class="text-center">Délka vlny: <span id="canvas-wavelength"><?php echo number_format($wavelengthCm, 2); ?> cm</span></p>
                            <div id="wave-visualization">
                                <svg width="100%" height="100%" viewBox="0 0 1000 150" preserveAspectRatio="none">
                                    <defs>
                                        <linearGradient id="wave-gradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                            <stop offset="0%" stop-color="#007bff" />
                                            <stop offset="100%" stop-color="#00bfff" />
                                        </linearGradient>
                                        <filter id="glow">
                                            <feGaussianBlur stdDeviation="2" result="blur" />
                                            <feComposite in="SourceGraphic" in2="blur" operator="over" />
                                        </filter>
                                    </defs>
                                    <?php
                                    // Generování SVG cesty pro sinusovou vlnu
                                    $width = 1000;
                                    $height = 150;
                                    $center = $height / 2;
                                    $amplitude = 50;
                                    // Výpočet vlnové délky v pixelech - frekvence je v MHz, což je správná jednotka pro tento výpočet
                                    $wavelengthPixels = 150 * pow(145 / max(1, $frequency), 0.7);

                                    $pathData = "M0,$center";
                                    for ($x = 0; $x <= $width; $x += 3) {
                                        $mainWave = sin(($x / $wavelengthPixels) * 2 * M_PI);
                                        $harmonic = 0.15 * sin(($x / $wavelengthPixels) * 4 * M_PI);
                                        $combinedWave = $mainWave + $harmonic;
                                        $y = $center + $amplitude * $combinedWave;
                                        $pathData .= " L$x,$y";
                                    }
                                    ?>
                                    <path d="<?php echo $pathData; ?>" stroke="url(#wave-gradient)" stroke-width="4" fill="none" filter="url(#glow)" />
                                </svg>
                                <div id="wave-info"><?php echo $frequency; ?> MHz</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="row mt-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-info text-white">
                            <h3 class="card-title">Informace o rádiových pásmech</h3>
                        </div>
                        <div class="card-body">
                            <table class="table table-dark">
                                <thead>
                                    <tr class="bg-primary">
                                        <th>Označení</th>
                                        <th>Frekvenční rozsah</th>
                                        <th>Typické využití</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td>ELF</td>
                                        <td>3-30 Hz</td>
                                        <td>Komunikace s ponorkami</td>
                                    </tr>
                                    <tr>
                                        <td>SLF</td>
                                        <td>30-300 Hz</td>
                                        <td>Komunikace s ponorkami, energetické sítě</td>
                                    </tr>
                                    <tr>
                                        <td>ULF</td>
                                        <td>300-3000 Hz</td>
                                        <td>Komunikace v dolech</td>
                                    </tr>
                                    <tr>
                                        <td>VLF</td>
                                        <td>3-30 kHz</td>
                                        <td>Námořní navigace, časové signály</td>
                                    </tr>
                                    <tr>
                                        <td>LF</td>
                                        <td>30-300 kHz</td>
                                        <td>Navigace, časové signály, AM vysílání</td>
                                    </tr>
                                    <tr>
                                        <td>MF</td>
                                        <td>300-3000 kHz</td>
                                        <td>AM vysílání, námořní komunikace</td>
                                    </tr>
                                    <tr>
                                        <td>HF</td>
                                        <td>3-30 MHz</td>
                                        <td>Krátké vlny, dálková komunikace</td>
                                    </tr>
                                    <tr>
                                        <td>VHF</td>
                                        <td>30-300 MHz</td>
                                        <td>FM vysílání, TV, letecká komunikace</td>
                                    </tr>
                                    <tr>
                                        <td>UHF</td>
                                        <td>300-3000 MHz</td>
                                        <td>TV, mobilní telefony, Wi-Fi</td>
                                    </tr>
                                    <tr>
                                        <td>SHF</td>
                                        <td>3-30 GHz</td>
                                        <td>Mikrovlnné spoje, satelitní komunikace</td>
                                    </tr>
                                    <tr>
                                        <td>EHF</td>
                                        <td>30-300 GHz</td>
                                        <td>Radar, radioastronomie</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <footer class="bg-dark text-white text-center text-lg-start mt-4">
        <div class="container p-4">
            <div class="row">
                <div class="col-lg-6 col-md-12 mb-4 mb-md-0">
                    <h5>O projektu AllHam</h5>
                    <p>
                        AllHam je komplexní nástroj pro radioamatéry, který poskytuje různé výpočty a informace užitečné pro radioamatérskou činnost.
                    </p>
                </div>
                <div class="col-lg-6 col-md-12 mb-4 mb-md-0">
                    <h5>Kontakt</h5>
                    <p>
                        <a href="https://www.OK2HSS.cz" class="text-info">www.ok2hss.cz</a>
                    </p>
                </div>
            </div>
        </div>
        <div class="text-center p-3" style="background-color: rgba(0, 0, 0, 0.2);">
            © 2023 AllHam
        </div>
    </footer>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <!-- Dark Mode JS -->
    <script src="js/dark-mode.js"></script>
</body>
</html>
