/**
 * 3D Print Calculator
 * Modern JavaScript implementation with animations
 */

// Vue application
const app = new Vue({
    el: '#app',
    data: {
        // Input values
        printName: '',
        printHours: 0,
        printMinutes: 0,
        materialWeight: 0,
        hourlyRate: 25,
        materialPrice: 700,
        electricityPrice: 6.5, // Cena elektriny v Kč/kWh
        printerPower: 0.3, // Príkon tlačiarne v kW (predvolená hodnota)

        // Results
        materialCost: 0,
        printingCost: 0,
        electricityCost: 0,
        totalCost: 0,

        // UI states
        calculating: false,
        resultsVisible: false,
        animateResults: false,
        nameError: '',

        // History
        calculationsHistory: []
    },
    methods: {
        /**
         * Calculate all costs
         */
        calculate() {
            // Reset previous error
            this.nameError = '';

            // Validate required fields
            if (!this.printName || this.printName.trim() === '') {
                this.nameError = 'Jméno tisku je povinné pole!';
                return;
            }

            // Start calculation animation
            this.calculating = true;
            this.animateButton();

            // Prepare form data for AJAX request
            const formData = new FormData();
            formData.append('ajax', 'true');
            formData.append('action', 'calculate');
            formData.append('printName', this.printName.trim());
            formData.append('hours', parseFloat(this.printHours) || 0);
            formData.append('minutes', parseFloat(this.printMinutes) || 0);
            formData.append('weight', parseFloat(this.materialWeight) || 0);
            formData.append('hourlyRate', parseFloat(this.hourlyRate) || 0);
            formData.append('materialPrice', parseFloat(this.materialPrice) || 0);
            formData.append('electricityPrice', parseFloat(this.electricityPrice) || 0);
            formData.append('printerPower', parseFloat(this.printerPower) || 0);

            // Send AJAX request to server
            fetch('index.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    // Handle server-side validation error
                    this.nameError = data.error;
                    this.calculating = false;
                    return;
                }

                // Update results from server response
                this.materialCost = data.materialCost;
                this.printingCost = data.printingCost;
                this.electricityCost = data.electricityCost;
                this.totalCost = data.totalCost;

                // Show results with animation
                this.resultsVisible = true;
                this.calculating = false;
                this.animateResults = true;

                // Load updated history
                this.loadHistory();

                // Reset animation flag after animation completes
                setTimeout(() => {
                    this.animateResults = false;
                }, 500);
            })
            .catch(error => {
                console.error('Error:', error);
                this.calculating = false;
                alert('Došlo k chybě při výpočtu. Zkuste to prosím znovu.');
            });
        },

        /**
         * Animate the calculate button
         */
        animateButton() {
            const button = document.querySelector('.btn-calculate');
            button.classList.add('pulse');

            // Remove animation class after animation completes
            setTimeout(() => {
                button.classList.remove('pulse');
            }, 500);
        },

        /**
         * Format number as currency
         */
        formatCurrency(value) {
            return value.toLocaleString('cs-CZ') + ' Kč';
        },

        /**
         * Load calculations history from server
         */
        loadHistory() {
            const formData = new FormData();
            formData.append('ajax', 'true');
            formData.append('action', 'loadHistory');

            fetch('index.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                this.calculationsHistory = data.calculations || [];
            })
            .catch(error => {
                console.error('Error loading history:', error);
            });
        },

        /**
         * Load calculation data from history item
         */
        loadCalculation(calc) {
            this.printName = calc.printName;
            this.printHours = calc.hours;
            this.printMinutes = calc.minutes;
            this.materialWeight = calc.weight;
            this.hourlyRate = calc.hourlyRate;
            this.materialPrice = calc.materialPrice;
            this.electricityPrice = calc.electricityPrice;
            this.printerPower = calc.printerPower;

            // Clear any previous results and errors
            this.resultsVisible = false;
            this.nameError = '';

            // Scroll to top of form
            document.querySelector('.calculator-container').scrollIntoView({
                behavior: 'smooth'
            });
        },

        /**
         * Format date for display
         */
        formatDate(timestamp) {
            const date = new Date(timestamp);
            return date.toLocaleString('cs-CZ', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        }
    },

    /**
     * Load history when component is mounted
     */
    mounted() {
        this.loadHistory();
    }
});

/**
 * Initialize tooltips and other Bootstrap components
 */
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Bootstrap tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function(tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
});
